import os
import csv
import cv2
import tkinter as tk
from tkinter import filedialog, simpledialog, messagebox


def pick_video_file():
    root = tk.Tk()
    root.withdraw()
    path = filedialog.askopenfilename(
        title="Select video file",
        filetypes=[
            ("Video files", "*.mp4 *.avi *.mov *.mkv *.m4v *.mpg *.mpeg *.wmv"),
            ("All files", "*.*"),
        ],
    )
    root.destroy()
    return path


def pick_output_folder():
    root = tk.Tk()
    root.withdraw()
    folder = filedialog.askdirectory(title="Select output folder for frames")
    root.destroy()
    return folder


def ask_every_n_frames(default=5):
    root = tk.Tk()
    root.withdraw()
    n = simpledialog.askinteger(
        "Frame step",
        f"Extract every N-th frame (default {default})",
        initialvalue=default,
        minvalue=1,
        maxvalue=10_000,
    )
    root.destroy()
    return n


def main():
    video_path = pick_video_file()
    if not video_path:
        print("No video selected. Exiting.")
        return

    out_dir = pick_output_folder()
    if not out_dir:
        print("No output folder selected. Exiting.")
        return

    every_n = ask_every_n_frames(default=5)
    if every_n is None:
        print("Cancelled. Exiting.")
        return

    cap = cv2.VideoCapture(video_path)
    if not cap.isOpened():
        messagebox.showerror("Error", f"Could not open video:\n{video_path}")
        return

    fps = cap.get(cv2.CAP_PROP_FPS)
    total_frames = int(cap.get(cv2.CAP_PROP_FRAME_COUNT) or 0)

    base = os.path.splitext(os.path.basename(video_path))[0]
    frames_folder = os.path.join(out_dir, f"{base}_frames_every{every_n}")
    os.makedirs(frames_folder, exist_ok=True)

    index_csv = os.path.join(frames_folder, "frames_index.csv")

    saved = 0
    i = 0

    with open(index_csv, "w", newline="") as f:
        w = csv.writer(f)
        w.writerow(["saved_filename", "frame_index", "time_s", "fps", "source_video"])

        while True:
            ok, frame = cap.read()
            if not ok:
                break

            if i % every_n == 0:
                t = (i / fps) if fps and fps > 0 else ""
                fname = f"frame_{i:06d}.jpg"
                out_path = os.path.join(frames_folder, fname)

                # Save JPG (quality 95). Change to PNG if you prefer lossless.
                cv2.imwrite(out_path, frame, [int(cv2.IMWRITE_JPEG_QUALITY), 95])

                w.writerow([fname, i, t, fps, os.path.basename(video_path)])
                saved += 1

            i += 1

    cap.release()

    msg = (
        f"Done!\n\n"
        f"Video: {video_path}\n"
        f"FPS: {fps}\n"
        f"Total frames (reported): {total_frames}\n"
        f"Extracted every {every_n} frame(s)\n"
        f"Saved frames: {saved}\n\n"
        f"Output folder:\n{frames_folder}\n"
        f"Index file:\n{index_csv}"
    )
    messagebox.showinfo("Finished", msg)
    print(msg)


if __name__ == "__main__":
    # Requires: pip install opencv-python
    main()
