import os
import glob
import csv
import numpy as np

import tkinter as tk
from tkinter import filedialog, messagebox

import matplotlib.pyplot as plt
from matplotlib.image import imread


# ---------------------------
# Angle convention utilities
# ---------------------------

def angle_up0_clockwise(dx, dy):
    """
    Image coordinates: x to the right, y down.
    Returns degrees where:
      up = 0°, right = 90°, down = 180°, left = 270°
    """
    ang = (np.degrees(np.arctan2(dx, -dy)) + 360.0) % 360.0
    return float(ang)

def angular_diff_clockwise(from_deg, to_deg):
    """
    Clockwise rotation needed to go from 'from_deg' to 'to_deg', in [0, 360).
    Example: from=10, to=40 -> 30
             from=350, to=10 -> 20
    """
    return (to_deg - from_deg) % 360.0


# ---------------------------
# Frame loading
# ---------------------------

def choose_folder():
    root = tk.Tk()
    root.withdraw()
    folder = filedialog.askdirectory(title="Select folder with extracted frames")
    root.destroy()
    return folder

def list_images(folder):
    exts = ("*.png", "*.jpg", "*.jpeg", "*.tif", "*.tiff", "*.bmp")
    files = []
    for e in exts:
        files.extend(glob.glob(os.path.join(folder, e)))
    files = sorted(files)
    return files


# ---------------------------
# Interactive annotator
# ---------------------------

class ArrowAnnotator:
    """
    Click tail, then click head to define arrow.
    Keys:
      n / enter : accept current arrow (when two clicks are done)
      r         : redo arrow on this frame
      b         : go back one frame
      s         : skip this frame (angle = NaN)
      q / esc   : quit (saves what you have)
    """
    def __init__(self, image_files):
        self.files = image_files
        self.N = len(image_files)
        self.idx = 0

        # results
        self.north_angle = None  # absolute angle of the "north" arrow from first frame
        self.rows = {}  # idx -> dict with results

        # current arrow clicks
        self.clicks = []  # [(x,y), (x,y)]

        # matplotlib state
        self.fig, self.ax = plt.subplots()
        self.im_artist = None
        self.arrow_artist = None
        self.text_artist = None

        self.cid_click = self.fig.canvas.mpl_connect("button_press_event", self.on_click)
        self.cid_key = self.fig.canvas.mpl_connect("key_press_event", self.on_key)

        self.update_title()
        self.load_frame()

    def update_title(self):
        base = os.path.basename(self.files[self.idx])
        if self.idx == 0 and self.north_angle is None:
            self.fig.suptitle(
                f"[1/ {self.N}] SET NORTH on first frame: click TAIL then HEAD. File: {base}\n"
                "Keys: Enter/n=accept, r=redo, q/esc=quit",
                fontsize=10
            )
        else:
            north_txt = "unset" if self.north_angle is None else f"{self.north_angle:.2f}°"
            self.fig.suptitle(
                f"[{self.idx+1}/ {self.N}] Draw butterfly heading (TAIL → HEAD). File: {base}\n"
                f"North reference: {north_txt} | Keys: Enter/n=accept, r=redo, b=back, s=skip, q/esc=quit",
                fontsize=10
            )

    def load_frame(self):
        self.ax.clear()
        img = imread(self.files[self.idx])
        self.im_artist = self.ax.imshow(img)
        self.ax.set_axis_off()

        self.clicks = []
        self.clear_overlay()

        # If already annotated, display it
        if self.idx in self.rows and self.rows[self.idx].get("status") == "ok":
            tail = (self.rows[self.idx]["tail_x"], self.rows[self.idx]["tail_y"])
            head = (self.rows[self.idx]["head_x"], self.rows[self.idx]["head_y"])
            self.draw_arrow(tail, head, color="lime")

            abs_ang = self.rows[self.idx]["angle_abs_deg"]
            rel_ang = self.rows[self.idx]["angle_rel_deg"]
            self.show_text(f"ABS: {abs_ang:.2f}°   REL: {rel_ang:.2f}°", color="lime")

        elif self.idx in self.rows and self.rows[self.idx].get("status") == "skip":
            self.show_text("SKIPPED", color="yellow")

        self.update_title()
        self.fig.canvas.draw_idle()

    def clear_overlay(self):
        if self.arrow_artist is not None:
            self.arrow_artist.remove()
            self.arrow_artist = None
        if self.text_artist is not None:
            self.text_artist.remove()
            self.text_artist = None

    def draw_arrow(self, tail, head, color="red"):
        self.clear_overlay()
        x0, y0 = tail
        x1, y1 = head
        self.arrow_artist = self.ax.annotate(
            "",
            xy=(x1, y1), xytext=(x0, y0),
            arrowprops=dict(arrowstyle="->", linewidth=2, color=color)
        )

    def show_text(self, text, color="red"):
        self.text_artist = self.ax.text(
            0.02, 0.02, text,
            transform=self.ax.transAxes,
            fontsize=12, color=color,
            bbox=dict(facecolor="black", alpha=0.5, edgecolor="none", pad=6)
        )

    def on_click(self, event):
        if event.inaxes != self.ax:
            return
        if event.xdata is None or event.ydata is None:
            return

        # left click only
        if event.button != 1:
            return

        self.clicks.append((float(event.xdata), float(event.ydata)))

        if len(self.clicks) == 1:
            self.draw_arrow(self.clicks[0], self.clicks[0], color="red")
            self.show_text("Click HEAD", color="red")

        elif len(self.clicks) == 2:
            tail, head = self.clicks
            self.draw_arrow(tail, head, color="red")

            dx = head[0] - tail[0]
            dy = head[1] - tail[1]
            ang_abs = angle_up0_clockwise(dx, dy)

            if self.idx == 0 and self.north_angle is None:
                # first frame: define north reference
                self.show_text(f"North set candidate: {ang_abs:.2f}°  (press Enter to accept)", color="cyan")
            else:
                if self.north_angle is None:
                    self.show_text("North is not set (should be set on frame 1).", color="orange")
                else:
                    ang_rel = angular_diff_clockwise(self.north_angle, ang_abs)
                    self.show_text(f"ABS: {ang_abs:.2f}°   REL: {ang_rel:.2f}°  (press Enter)", color="red")

        else:
            # more than 2 clicks -> keep only last two
            self.clicks = self.clicks[-2:]

        self.fig.canvas.draw_idle()

    def accept_current(self):
        if len(self.clicks) != 2:
            return

        tail, head = self.clicks
        dx = head[0] - tail[0]
        dy = head[1] - tail[1]
        ang_abs = angle_up0_clockwise(dx, dy)

        # First frame: set north if not yet set
        if self.idx == 0 and self.north_angle is None:
            self.north_angle = ang_abs
            ang_rel = 0.0
            status = "north"
            color = "cyan"
            msg = f"NORTH ACCEPTED: {self.north_angle:.2f}°"
        else:
            if self.north_angle is None:
                messagebox.showerror("North not set", "North reference is not set. Please annotate the first frame first.")
                return
            ang_rel = angular_diff_clockwise(self.north_angle, ang_abs)
            status = "ok"
            color = "lime"
            msg = f"ABS: {ang_abs:.2f}°   REL: {ang_rel:.2f}°  (saved)"

        self.rows[self.idx] = dict(
            frame_index=self.idx,
            filename=os.path.basename(self.files[self.idx]),
            tail_x=tail[0], tail_y=tail[1],
            head_x=head[0], head_y=head[1],
            angle_abs_deg=ang_abs,
            angle_rel_deg=ang_rel,
            status=status
        )

        self.draw_arrow(tail, head, color=color)
        self.show_text(msg, color=color)
        self.fig.canvas.draw_idle()

        self.next_frame()

    def redo(self):
        self.clicks = []
        if self.idx in self.rows:
            del self.rows[self.idx]
        self.clear_overlay()
        self.show_text("Redo: click TAIL then HEAD", color="red")
        self.fig.canvas.draw_idle()

    def skip(self):
        # allow skip even if north frame is first, but you really shouldn't
        self.rows[self.idx] = dict(
            frame_index=self.idx,
            filename=os.path.basename(self.files[self.idx]),
            tail_x=np.nan, tail_y=np.nan,
            head_x=np.nan, head_y=np.nan,
            angle_abs_deg=np.nan,
            angle_rel_deg=np.nan,
            status="skip"
        )
        self.clear_overlay()
        self.show_text("SKIPPED", color="yellow")
        self.fig.canvas.draw_idle()
        self.next_frame()

    def next_frame(self):
        if self.idx < self.N - 1:
            self.idx += 1
            self.load_frame()
        else:
            messagebox.showinfo("Done", "Last frame reached. Press q to quit and save.")
            self.update_title()
            self.fig.canvas.draw_idle()

    def back(self):
        if self.idx > 0:
            self.idx -= 1
            self.load_frame()

    def on_key(self, event):
        k = (event.key or "").lower()

        if k in ("enter", "n"):
            self.accept_current()

        elif k == "r":
            self.redo()

        elif k == "b":
            self.back()

        elif k == "s":
            self.skip()

        elif k in ("q", "escape"):
            plt.close(self.fig)

    def export_csv(self, out_csv_path):
        # make a row for every frame in order
        header = [
            "frame_index", "filename",
            "tail_x", "tail_y", "head_x", "head_y",
            "angle_abs_deg", "angle_rel_deg",
            "status"
        ]
        with open(out_csv_path, "w", newline="") as f:
            w = csv.DictWriter(f, fieldnames=header)
            w.writeheader()
            for i in range(self.N):
                if i in self.rows:
                    w.writerow(self.rows[i])
                else:
                    # unannotated
                    w.writerow(dict(
                        frame_index=i,
                        filename=os.path.basename(self.files[i]),
                        tail_x=np.nan, tail_y=np.nan,
                        head_x=np.nan, head_y=np.nan,
                        angle_abs_deg=np.nan,
                        angle_rel_deg=np.nan,
                        status="missing"
                    ))


def main():
    folder = choose_folder()
    if not folder:
        print("No folder selected. Exiting.")
        return

    images = list_images(folder)
    if not images:
        messagebox.showerror("No images found", f"No image files found in:\n{folder}")
        return

    annot = ArrowAnnotator(images)
    plt.show()  # blocks until window closed

    out_csv = os.path.join(folder, "arrow_angles.csv")
    annot.export_csv(out_csv)

    # Report
    total = len(images)
    ok = sum(1 for v in annot.rows.values() if v.get("status") in ("ok", "north"))
    skipped = sum(1 for v in annot.rows.values() if v.get("status") == "skip")
    print(f"Saved: {out_csv}")
    print(f"Annotated: {ok}/{total}, Skipped: {skipped}/{total}")


if __name__ == "__main__":
    main()
