import cv2
import tkinter as tk
from tkinter import filedialog

# ======================================
# Step 0 — Choose video using file dialog
# ======================================

# Hide main Tk window
root = tk.Tk()
root.withdraw()

video_path = filedialog.askopenfilename(
    title="Select input video",
    filetypes=[("Video files", "*.mp4 *.avi *.mov *.mkv"), ("All files", "*.*")]
)

if not video_path:
    raise ValueError("No video selected. Exiting.")

print("Selected video:", video_path)

output_path = video_path.replace(".mp4", "_cropped.mp4")

# ======================================
# Step 1 — Load first frame
# ======================================
cap = cv2.VideoCapture(video_path)
ret, first_frame = cap.read()
if not ret:
    raise ValueError("Cannot read first frame from the video.")

clone = first_frame.copy()
crop_rect = None

# ======================================
# Step 2 — Mouse callback to select crop
# ======================================
drawing = False
ix, iy = -1, -1

def draw_rectangle(event, x, y, flags, param):
    global ix, iy, drawing, crop_rect, clone

    temp = clone.copy()

    if event == cv2.EVENT_LBUTTONDOWN:
        drawing = True
        ix, iy = x, y

    elif event == cv2.EVENT_MOUSEMOVE:
        if drawing:
            cv2.rectangle(temp, (ix, iy), (x, y), (0, 255, 0), 2)
            cv2.imshow("Select Crop Area", temp)

    elif event == cv2.EVENT_LBUTTONUP:
        drawing = False
        crop_rect = (min(ix, x), min(iy, y), abs(x - ix), abs(y - iy))
        cv2.rectangle(temp, (ix, iy), (x, y), (0, 255, 0), 2)
        cv2.imshow("Select Crop Area", temp)
        print("Selected crop:", crop_rect)

# ======================================
# Step 3 — Show selection window
# ======================================
cv2.namedWindow("Select Crop Area")
cv2.setMouseCallback("Select Crop Area", draw_rectangle)
cv2.imshow("Select Crop Area", clone)

print("Draw rectangle with mouse. Press ENTER to confirm, ESC to cancel.")

while True:
    key = cv2.waitKey(1) & 0xFF
    if key == 13:  # ENTER
        break
    elif key == 27:  # ESC
        print("Cancelled.")
        cap.release()
        cv2.destroyAllWindows()
        exit()

cv2.destroyAllWindows()

if crop_rect is None:
    raise ValueError("No crop selection made.")

x, y, w, h = crop_rect
print(f"Cropping using: x={x}, y={y}, w={w}, h={h}")

# ======================================
# Step 4 — Crop entire video
# ======================================
cap = cv2.VideoCapture(video_path)
fps = cap.get(cv2.CAP_PROP_FPS)

fourcc = cv2.VideoWriter_fourcc(*"mp4v")
out = cv2.VideoWriter(output_path, fourcc, fps, (w, h))

while True:
    ret, frame = cap.read()
    if not ret:
        break

    crop = frame[y:y+h, x:x+w]
    out.write(crop)

cap.release()
out.release()

print("Saved cropped video:", output_path)
